package furny.swing.admin;

import java.awt.GridBagConstraints;
import java.awt.GridBagLayout;
import java.awt.Insets;
import java.util.concurrent.Callable;
import java.util.logging.Logger;

import javax.swing.JFrame;
import javax.swing.JPanel;
import javax.swing.JTabbedPane;

import com.jme3.system.AppSettings;

import furny.entities.Furniture;
import furny.furndb.importer.DBModelSource;
import furny.jme.FurnyApplication;
import furny.swing.admin.importer.ImporterPanel;
import furny.swing.admin.tags.TagPanel;
import furny.swing.admin.viewer.IFurnitureViewer;
import furny.swing.admin.viewer.IModelViewer;
import furny.swing.admin.viewer.ModelViewerState;
import furny.swing.admin.viewer.ViewerCanvasPanel;

/**
 * Admin main frame for the furniture database.
 * 
 * @since 12.08.2012
 * @author Stephan Dreyer
 */
@SuppressWarnings("serial")
public class FurnDBAdminFrame extends JFrame implements IFurnitureViewer {
  private static final Logger LOGGER = Logger.getLogger(FurnDBAdminFrame.class
      .getName());

  private BrowserPanel browserPanel;

  private final JTabbedPane tabbedPane;
  private transient final IModelViewer viewer;
  private final FurnyApplication app;
  private final ViewerCanvasPanel viewerPanel;
  private final ImporterPanel importerPanel;

  /**
   * Instantiates a new furn db admin frame.
   * 
   * @since 12.08.2012
   * @author Stephan Dreyer
   */
  public FurnDBAdminFrame() {
    super("FurnDBAdmin");
    setDefaultCloseOperation(EXIT_ON_CLOSE);

    setLayout(new GridBagLayout());
    final GridBagConstraints constraints = new GridBagConstraints();
    constraints.gridx = 0;
    constraints.gridy = 0;
    constraints.fill = GridBagConstraints.BOTH;
    constraints.weightx = 1.0d;
    constraints.weighty = 1.0d;

    tabbedPane = new JTabbedPane();

    this.app = new FurnyApplication();

    final AppSettings settings = new AppSettings(true);
    settings.setRenderer(AppSettings.LWJGL_OPENGL2);
    settings.setTitle(getTitle());
    app.setSettings(settings);

    final ModelViewerState state = new ModelViewerState(new DBModelSource(),
        app);
    this.viewer = state;

    app.enqueue(new Callable<Void>() {
      @Override
      public Void call() throws Exception {
        app.getStateManager().attach(state);
        return null;
      }
    });

    app.setPauseOnLostFocus(false);
    app.setShowSettings(false);

    LOGGER.fine(app.getAppSettings().toString());

    viewerPanel = new ViewerCanvasPanel(app, viewer);

    importerPanel = new ImporterPanel(app);

    // /////////////////////////////////////////////////

    tabbedPane.addTab("Furnitures", createFurniturePanel());
    tabbedPane.addTab("Viewer", viewerPanel);
    tabbedPane.addTab("Tags", createTagsPanel());
    tabbedPane.addTab("Import", importerPanel);

    add(tabbedPane, constraints);

    setSize(800, 600);
    setLocationRelativeTo(null);

    LOGGER.info("Starting up admin application");
  }

  /**
   * Creates the tags panel.
   * 
   * @return the j panel
   * @since 12.08.2012
   * @author Stephan Dreyer
   */
  private JPanel createTagsPanel() {
    final JPanel panel = new JPanel();
    panel.setLayout(new GridBagLayout());
    final GridBagConstraints constraints = new GridBagConstraints();
    constraints.insets = new Insets(5, 5, 5, 5);
    constraints.gridx = 0;
    constraints.gridy = 0;
    constraints.fill = GridBagConstraints.BOTH;
    constraints.weightx = 1d;
    constraints.weighty = 1d;

    panel.add(new TagPanel(), constraints);

    return panel;
  }

  @Override
  public void setFurniture(final Furniture furn) {
    viewer.setFurniture(furn);
    if (furn != null) {
      viewerPanel.updateViewerSize();
      tabbedPane.setSelectedComponent(viewerPanel);
    }
  }

  /**
   * Creates the furniture panel.
   * 
   * @return the j panel
   * @since 12.08.2012
   * @author Stephan Dreyer
   */
  private JPanel createFurniturePanel() {
    final JPanel panel = new JPanel();
    panel.setLayout(new GridBagLayout());
    final GridBagConstraints constraints = new GridBagConstraints();
    constraints.insets = new Insets(5, 5, 5, 5);
    constraints.gridx = 0;
    constraints.gridy = 0;
    constraints.fill = GridBagConstraints.BOTH;
    constraints.weightx = 1d;
    constraints.weighty = 1d;

    browserPanel = new BrowserPanel(this);
    panel.add(browserPanel, constraints);

    return panel;
  }
}
